	program MeshGenRect
! Rectangular mesh generator
! assumptions: (1) material #=1, (2) cons Vi, Dij

	implicit none
	integer Nx, Ny, Ne, Nn, Nb, BCe, ie(5), Ns, Nxy(0:4), ierror
	integer Nxx, Nyy
	integer e, n, f, ix, iy, k, s
	real*8 Xmin, Xmax, Ymin, Ymax, Lx, Ly, dx, dy, Vx, Vy, Dxx, Dxy,Dyx, Dyy, x(2)
	logical Darcy					! T Darcy-2D, F for FEM-2D
	logical Dcorner
	logical delta					! T for delta(Xdelta, Ydelta)
	real*8 Xdelta, Ydelta
	real*8 Vf, Vn
	character*1 BCt(4)				! south, east, north, west

	real*8, allocatable :: C(:), BCv(:,:,:), xy(:,:), vl(:,:,:)
	real*8, allocatable :: xx(:), yy(:), fx(:,:), gy(:,:), Sxy(:,:)

	data n/0/, e/0/, ierror/0/

! write param file (part of finp.txt)
	open(3, file='param.txt', status='unknown')
	write(3,*) '**********************************'	! version ID
	write(3,*) 'MeshGenRect v54.3 of 12/09/18'	! version ID
	write(3,*) '**********************************'	! version ID

! read MeshGenRect input file
	open(1, file='meshinp.txt', status='old')
	read(1,*) Darcy	
	read(1,*) Xmin, Xmax, Ymin, Ymax
	if(Darcy) then
	  read(1,*) Nx, Ny
	  Ns = 1	! Ns=1 used for Darcy
	else
	  read(1,*) Nx, Ny, Ns
	endif

	allocate ( C(Ns), BCv(4,Ns,2) )

	if(.not. Darcy) then
	  read(1,*) Vx, Vy
	endif
	read(1,*) Dxx, Dxy, Dyx, Dyy

! an independent source is assumed to be given as S=f(x)*g(y) for each species
	read(1,*) Nxx, Nyy			! No. of x, y data points (must be >1)

	if(Nxx.lt.2 .or. Nyy.lt.2) then
	  write(3,*) '!!! ERROR !!! Nxx or Nyy <2 !'
	  stop
	endif

	allocate ( xx(Nxx), fx(Ns,Nxx), yy(Nyy), gy(Ns,Nyy), Sxy(4,Ns) )

! f(x)
	read(1,*) (xx(k), k=1,Nxx)	! coordinate x
	do s = 1,Ns
	  read(1,*) (fx(s,k), k=1,Nxx)	! f(x) value
	enddo	! s
! g(y)
	read(1,*) (yy(k), k=1,Nyy)	! coordinate y
	do s = 1,Ns
	  read(1,*) (gy(s,k), k=1,Nyy)	! g(y) value
	enddo	! s

	if(xx(1).ne.Xmin .or. xx(Nxx).ne.Xmax .or. &
           yy(1).ne.Ymin .or. yy(Nyy).ne.Ymax) then
	  write(3,*) '!!! ERROR !!! xx(1).ne.Xmin .or. xx(Nxx).ne.Xmax .or.', &
	 	'yy(1).ne.Ymin .or. yy(Nyy).ne.Ymax !'
	  stop
	endif

	read(1,*) delta, Xdelta, Ydelta
	read(1,*) (C(s), s=1,Ns)
	read(1,*) (BCt(f), f=1,4)
	read(1,*) (Nxy(f), f=1,4)			! No. of data points (must be >1)

	if(Nxy(1).lt.2 .or.Nxy(2).lt.2 .or.Nxy(3).lt.2 .or.Nxy(4).lt.2) then
	  write(3,*) '!!! ERROR !!! Nxy(f) <2 !'
	  stop
	endif

	Nxy(0) = max(Nxy(1),Nxy(2),Nxy(3),Nxy(4))	! max. nodes in each face
	allocate ( xy(4,Nxy(0)), vl(4,Ns,Nxy(0)) )

	do f=1,4
	  read(1,*) (xy(f,k), k=1,Nxy(f))	! coordinate (x or y)
	  do s = 1,Ns
	    read(1,*) (vl(f,s,k), k=1,Nxy(f))	! value
	  enddo	! s
	enddo	! f
	close(1)

	if(xy(1,1).ne.Xmin .or. xy(1,Nxy(1)).ne.Xmax .or. &
	   xy(2,1).ne.Ymin .or. xy(2,Nxy(2)).ne.Ymax .or. &
	   xy(3,1).ne.Xmin .or. xy(3,Nxy(3)).ne.Xmax .or. &
           xy(4,1).ne.Ymin .or. xy(4,Nxy(4)).ne.Ymax) then
	  write(3,*) '!!! ERROR !!! xy(f,1).ne.Xmin or Ymin .or. xx(f,Nxy(f)).ne.Xmax or Ymax!'
	  stop
	endif

	Lx = Xmax-Xmin	
	Ly = Ymax-Ymin	
	Ne = Nx*Ny
	Nn = (Nx+1)*(Ny+1)
	Nb = 2*(Nx+Ny)
	dx = Lx/Nx
	dy = Ly/Ny

	write(3,'(5i10)')  Ne,  Nn,  Nb,   1, Ns
	write(3,'(a   )') 'Ne,  Nn,  Nb,  Nm, Ns'

! write nodes file
	open(1, file='nodes.txt', status='unknown')

	do iy = 1, Ny+1
	  do ix = 1, Nx+1
	    n = n+1
	    x(1) = dx * (ix-1) + Xmin
	    x(2) = dy * (iy-1) + Ymin
! Co = delta(x,y)
	    if(delta)	then
	      if(abs(x(1)-Xdelta).lt.0.6*dx .and. abs(x(2)-Ydelta).lt.0.6*dy) then
		write(1,'(i10, 2g14.6, 9g12.4)') n, x(1), x(2), (C(s), s = 1,Ns)
	      else
		write(1,'(i10, 2g14.6, 9g12.4)') n, x(1), x(2), (0.  , s = 1,Ns)
	      endif
	    else
! Co = const
	      write(1,'(i10 ,2g14.6 ,9g12.4)') n, x(1), x(2), (C(s), s = 1,Ns)
	    endif
	  enddo	! ix
	enddo	! iy
	write(1,'(a)') 'n, x, y, c(x,y|0, 1:Ns)'
	close(1)

! write elements and VD files
! write independent source file
	open(1, file='elements.txt', status='unknown')
	open(2, file=       'D.txt', status='unknown')
	open(4, file=  'source.txt', status='unknown')
	if(.not. Darcy)  open(5, file=       'V.txt', status='unknown')

	do iy = 1, Ny
	  do ix = 1, Nx
	    e = e+1
	    ie(1) = ix + (iy-1) * (Nx+1)
	    ie(2) = ie(1) + 1
	    ie(3) = ie(2) + 1 + Nx
	    ie(4) = ie(3) - 1
	    ie(5) = 1
	    write(1,'(i10, 5i10)') e, (ie(k), k=1,5)
	    write(2,'(i10, 4g12.4)') e, Dxx, Dxy, Dyx, Dyy
	    if(.not. Darcy)  write(5,'(i10, 2g12.4)') e, Vx, Vy

! 2D interpolation at all nodes of an element
	    call INTERPOL2(Nx, Ny, Ns, Xmin, Ymin, Nxx, Nyy, xx, yy, &
				fx, gy, dx, dy, ix, iy, Sxy, ierror)
	    do s = 1, Ns
	      write (4,'(2i10, 4g12.4)') e, s, (Sxy(k,s), k=1,4)
	    enddo	! s

	  enddo	! ix
	enddo	! iy
	write(1,'(a)') 'e, n1, n2, n3, n4, mat'
	close(1)
	write(2,'(a)') 'e, Dxx, Dxy, Dyx, Dyy'
	close(2)
	if(.not. Darcy) then
	  write(5,'(a)') 'e, Vx, Vy'
	  close(5)
	endif

	write (4,'(a)') 'e, s, (Sxy(k,s), k=1,4)'
	close(4)

! write BCs file
	open(1, file='BCs.txt', status='unknown')
	f = 1	! south (go in +x direction)
	do ix = 1, Nx
	  x(1) = dx * (ix-1) + Xmin	! x at node 1
	  x(2) = dx * (ix  ) + Xmin	! x at node 2
! 1D interpolation at all nodes of a boundary element face
	  call INTERPOL1(Ns, Nxy, f, xy, vl, dx, x, BCv, ierror)
	  BCe = ix
	  write(1,'(2i10, a5, 18g12.4)') f, BCe, BCt(f), &
		(BCv(f,s,1), BCv(f,s,2), s = 1,Ns)
	enddo	! ix

	f = 2	! east (go in +y direction)
	do iy = 1, Ny
	  x(1) = dy * (iy-1) + Ymin	! y at node 1
	  x(2) = dy * (iy  ) + Ymin	! y at node 2
! 1D interpolation at all nodes of a boundary element face
	  call INTERPOL1(Ns, Nxy, f, xy, vl, dy, x, BCv, ierror)
	  BCe = Nx * iy
	  write(1,'(2i10, a5, 18g12.4)') f, BCe, BCt(f), &
		(BCv(f,s,1), BCv(f,s,2), s = 1,Ns)
	enddo	! iy

	f = 3	! north (go in -x direction)
	do ix = Nx, 1, -1
	  x(1) = dx * (ix  ) + Xmin	! x at node 1
	  x(2) = dx * (ix-1) + Xmin	! x at node 2
! 1D interpolation at all nodes of a boundary element face
	  call INTERPOL1(Ns, Nxy, f, xy, vl, dx, x, BCv, ierror)
	  BCe = ix + Nx * (Ny-1)
	  write(1,'(2i10, a5, 18g12.4)') f, BCe, BCt(f), &
		(BCv(f,s,1), BCv(f,s,2), s = 1,Ns)
	enddo	! ix

	f = 4	! west (go in -y direction)
	do iy = Ny, 1, -1
	  x(1) = dy * (iy  ) + Ymin	! y at node 1
	  x(2) = dy * (iy-1) + Ymin	! y at node 2
! 1D interpolation at all nodes of a boundary element face
	  call INTERPOL1(Ns, Nxy, f, xy, vl, dy, x, BCv, ierror)
	  BCe = 1 + Nx * (iy-1)
	  write(1,'(2i10, a5, 18g12.4)') f, BCe, BCt(f), &
		(BCv(f,s,1), BCv(f,s,2), s = 1,Ns)
	enddo	! iy

	write(1,'(a)') 'f, BCe, BCtype, (BCv(f,s,1), BCv(f,s,2), s = 1,Ns)'
	close(1)

! check if Dirichlet BC corners have the same values
	do f = 1,4
	  n = f+1		! neighbour face
	  if(f .eq. 4)	n = 1	! allow for f periodity
 	  Dcorner = BCt(f) .eq. 'D' .and. BCt(n) .eq. 'D'
	  do s = 1, Ns
	    if     (f .eq. 1) then
	      Vf = vl(f,s,Nxy(1) )	! faces 1 & 2
	      Vn = vl(n,s,1      )
	    else if(f .eq. 2) then
	      Vf = vl(f,s,Nxy(2) )	! faces 2 & 3
	      Vn = vl(n,s,Nxy(3) )
	    else if(f .eq. 3) then
	      Vf = vl(f,s,1      )	! faces 3 & 4
	      Vn = vl(n,s,Nxy(4) )
	    else
	      Vf = vl(f,s,1      )	! faces 4 & 1
	      Vn = vl(n,s,1      )
	    endif
	    if( Dcorner .and. Vf.ne.Vn) then	! Dirichlet BC corner
	      write (3,*) '!!! error !!! inconsistent Dirichlet BC corner values'
	      write (3,*) ' species s, face f and neighbour face n: s, f, n = ', s, f, n
	      write (3,*) ' f- and n- face values, Vf, Vn  = ', Vf, Vn
	      ierror = ierror + 1
	    endif
	  enddo		! s
	enddo		! f

! check for errors
	if(ierror .ne. 0)	then
	  write (3,*) '!!! ABORTED !!!'
	  stop
	endif
	close(3)

	end
